function veh = createConfiguration(vehConf, libraryFolder)
%createConfiguration Create vehicle data for a configuration
%
% createConfiguration(libraryFolder, vehConf) loads the components whose
%   names are in vehConf from the specified library
%
%   libraryFolder is the path to the library folder.
%
%   vehConf is a structure containing the name of the components to be
%       loaded from the library.

arguments
    vehConf
    libraryFolder {mustBeText} = 'library';
end

veh.vehType = vehConf.vehicle_type;

veh.body = load(fullfile(libraryFolder, 'vehicle_body', vehConf.vehicle_name));
veh.wh = load(fullfile(libraryFolder, 'wheels', vehConf.wheels_name));
veh.fd = load(fullfile(libraryFolder, 'final_drive', vehConf.finalDrive_name));
veh.gb = load(fullfile(libraryFolder, 'transmission', vehConf.transmission_name));

pwtMass = veh.wh.mass + veh.fd.mass + veh.gb.mass;

switch veh.vehType
    case 'cv'
        veh.eng = load(fullfile(libraryFolder, 'engine', vehConf.engine_name));
        pwtMass = pwtMass + veh.eng.mass;
    case 'bev'
        veh.em = load(fullfile(libraryFolder, 'e-machine', vehConf.em_name));
        veh.batt = load(fullfile(libraryFolder, 'battery', vehConf.battery_name));
        pwtMass = pwtMass + veh.em.mass + veh.batt.mass;
end
if isfield(vehConf, 'cap_name')
    veh.cap = load(fullfile(libraryFolder, 'supercapacitor', vehConf.cap_name));
    pwtMass = pwtMass + veh.cap.mass;
end

% Masses 
% Curb mass:
%   The user defines either curbMass or chassisMass
%       - If curbMass is specified, that is exactly the curb mass of the
%       vehicle and the componentsì weights are ignored. 
%       - If chassisMass is specified instead, the curb mass is the sum of
%       chassisMass and the components' weights.
% Load mass:
%   The user defines either maxLoad or grossWeight
%       - If maxLoad is specified, that is the maximum payload. 
%       - If grossWeight is specified, the maximum load weight is obtained
%       as the difference between (regulatory) gross weight and curb mass;
%       this is useful for HD applications

% Curb mass
if isfield(veh.body, 'curbMass') && isfield(veh.body, 'chassisMass')
    warning("The vehicle body defines both a curbMass and a " + ...
        "chassisMass. chassisMass will be ignored.")
elseif isfield(veh.body, 'chassisMass')
    veh.body.curbMass = veh.body.chassisMass + pwtMass;
end

% Add load mass to the vehicle body mass
if isfield(veh.body, 'maxLoad') && isfield(veh.body, 'grossWeight')
    warning("The vehicle body defines both a maxLoad and a " + ...
        "grossWeight. grossWeight will be ignored.")
elseif isfield(veh.body, 'grossWeight')
    veh.body.maxLoad = veh.body.grossWeight - veh.body.curbMass;
end

end